<?php /** @noinspection PhpUndefinedClassInspection */
/**
 * Glider for Elementor
 * Glider create a flexible slider and stylish carousel slide-show for Elementor
 * Exclusively on https://1.envato.market/glider-elementor
 *
 * @encoding        UTF-8
 * @version         2.0.4
 * @copyright       (C) 2018 - 2022 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 **/

namespace Merkulove\GliderElementor;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

use Exception;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Icons_Manager;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Image_Size;
use Elementor\Plugin;
use Elementor\Repeater;
use Elementor\Core\Schemes\Typography;
use Elementor\Core\Schemes\Color;
use Elementor\Utils;
use Merkulove\GliderElementor\Unity\Plugin as UnityPlugin;

/** @noinspection PhpUnused */
/**
 * Glider - Custom Elementor Widget.
 **/
class glider_elementor extends Widget_Base {

    /**
     * Use this to sort widgets.
     * A smaller value means earlier initialization of the widget.
     * Can take negative values.
     * Default widgets and widgets from 3rd party developers have 0 $mdp_order
     **/
    public $mdp_order = 1;

    /**
     * Widget base constructor.
     * Initializing the widget base class.
     *
     * @access public
     * @throws Exception If arguments are missing when initializing a full widget instance.
     * @param array      $data Widget data. Default is an empty array.
     * @param array|null $args Optional. Widget default arguments. Default is null.
     *
     * @return void
     **/
    public function __construct( $data = [], $args = null ) {

        parent::__construct( $data, $args );

        wp_register_style( 'mdp-glider-elementor-admin', UnityPlugin::get_url() . 'src/Merkulove/Unity/assets/css/elementor-admin' . UnityPlugin::get_suffix() . '.css', [], UnityPlugin::get_version() );
        wp_register_style( 'mdp-glider-elementor', UnityPlugin::get_url() . 'css/glider-elementor' . UnityPlugin::get_suffix() . '.css', [], UnityPlugin::get_version() );
	    wp_register_script( 'mdp-glider-elementor', UnityPlugin::get_url() . 'js/glider-elementor' . UnityPlugin::get_suffix() . '.js', [ 'jquery', 'elementor-frontend' ], UnityPlugin::get_version(), true );

    }

    /**
     * Return a widget name.
     *
     * @return string
     **/
    public function get_name() {

        return 'mdp-glider-elementor';

    }

    /**
     * Return the widget title that will be displayed as the widget label.
     *
     * @return string
     **/
    public function get_title() {

        return esc_html__( 'Glider', 'glider-elementor' );

    }

    /**
     * Set the widget icon.
     *
     * @return string
     */
    public function get_icon() {

        return 'mdp-glider-elementor-widget-icon';

    }

    /**
     * Set the category of the widget.
     *
     * @return array with category names
     **/
    public function get_categories() {

        return [ 'general' ];

    }

    /**
     * Get widget keywords. Retrieve the list of keywords the widget belongs to.
     *
     * @access public
     *
     * @return array Widget keywords.
     **/
    public function get_keywords() {

        return [ 'Merkulove', 'Glider', 'slider', 'carousel' ];

    }

    /**
     * Get style dependencies.
     * Retrieve the list of style dependencies the widget requires.
     *
     * @access public
     *
     * @return array Widget styles dependencies.
     **/
    public function get_style_depends() {

        return [ 'mdp-glider-elementor', 'mdp-glider-elementor-admin' ];

    }

	/**
	 * Get script dependencies.
	 * Retrieve the list of script dependencies the element requires.
	 *
	 * @access public
     *
	 * @return array Element scripts dependencies.
	 **/
	public function get_script_depends() {

		return [ 'mdp-glider-elementor' ];

    }

    /**
     * Add the widget controls.
     *
     * @since 1.0.0
     * @access protected
     *
     * @return void with category names
     **/
    protected function register_controls() {

        /** Content Tab. */
        if ( true ) {

            /** Items Section. */
            if ( true ) {

                $this->start_controls_section( 'section_items', ['label' => esc_html__( 'Items', 'glider-elementor' ), 'tab' => Controls_Manager::TAB_CONTENT] );

                $this->add_control(
                    'skin',
                    [
                        'label' => esc_html__( 'Skin', 'glider-elementor' ),
                        'type'  => Controls_Manager::SELECT,
                        'options' => [
                            'overlay'  => esc_html__( 'Overlay', 'glider-elementor' ),
                            'cards'   => esc_html__( 'Cards', 'glider-elementor' )
                        ],
                        'default' => 'overlay',
                        'separator' => 'none'
                    ]
                );

                # Content Position for Overlay
                $this->add_control(
                    'content_position',
                    [
                        'label' => esc_html__( 'Content Position', 'glider-elementor' ),
                        'type'  => Controls_Manager::SELECT,
                        'options' => [
                            'mdp-position-top-left'         => esc_html__( 'Top-Left', 'glider-elementor' ),
                            'mdp-position-top-center'       => esc_html__( 'Top-Center', 'glider-elementor' ),
                            'mdp-position-top-right'        => esc_html__( 'Top-Right', 'glider-elementor' ),
                            'mdp-position-center'           => esc_html__( 'Center', 'glider-elementor' ),
                            'mdp-position-center-left'      => esc_html__( 'Center-Left', 'glider-elementor' ),
                            'mdp-position-center-right'     => esc_html__( 'Center-Right', 'glider-elementor' ),
                            'mdp-position-bottom-left'      => esc_html__( 'Bottom-Left', 'glider-elementor' ),
                            'mdp-position-bottom-center'    => esc_html__( 'Bottom-Center', 'glider-elementor' ),
                            'mdp-position-bottom-right'     => esc_html__( 'Bottom-Right', 'glider-elementor' ),
                        ],
                        'default' => 'mdp-position-center',
                        'separator' => 'none',
                        'condition' => [
                            'skin' => 'overlay',
                        ]
                    ]
                );

                # Overlay Background
                $this->add_group_control(
                    Group_Control_Background::get_type(),
                    [
                        'name'              => 'overlay_background',
                        'types'             => [ 'classic', 'gradient' ],
                        'selector'          => '{{WRAPPER}} .mdp-slider-container .mdp-slider-items li > .mdp-cover-container:before',
                        'condition' => [
                            'skin' => 'overlay',
                        ]
                    ]
                );

                # Image Position for Cards
                $this->add_control(
                    'image_position',
                    [
                        'label' => esc_html__( 'Image Position', 'glider-elementor' ),
                        'type'  => Controls_Manager::SELECT,
                        'options' => [
                            'top'       => esc_html__( 'Top', 'glider-elementor' ),
                            'left'      => esc_html__( 'Left', 'glider-elementor' ),
                            'right'     => esc_html__( 'Right', 'glider-elementor' ),
                            'bottom'    => esc_html__( 'Bottom', 'glider-elementor' ),
                            'none'      => esc_html__( 'None', 'glider-elementor' ),
                        ],
                        'default'   => 'top',
                        'separator' => 'none',
                        'condition' => [
                            'skin'  => 'cards',
                        ]
                    ]
                );

                # Items Repeater
                $items = new Repeater();

                $items->add_control(
                    'image',
                    [
                        'label' => esc_html__( 'Choose Image', 'glider-elementor' ),
                        'type' => Controls_Manager::MEDIA,
                        'dynamic' => [
                            'active' => true,
                        ],
                        'default' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ]
                );

                $items->add_group_control(
                    Group_Control_Image_Size::get_type(),
                    [
                        'name' => 'image', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `image_size` and `image_custom_dimension`.
                        'default' => 'large',
                        'separator' => 'none',
                    ]
                );

                # Content
                $items->add_control(
                    'content_from',
                    [
                        'label' => esc_html__( 'Content', 'glider-elementor' ),
                        'type' => Controls_Manager::SELECT,
                        'options' => [
                            'none' => esc_html__( 'None', 'glider-elementor' ),
                            'attachment' => esc_html__( 'Attachment Details', 'glider-elementor' ),
                            'title' => esc_html__( 'Title', 'glider-elementor' ),
                            'custom' => esc_html__( 'Custom Content', 'glider-elementor' ),
                        ],
                        'default' => 'none'
                    ]
                );

                # Title
                $items->add_control(
                    'title',
                    [
                        'label'     => esc_html__( 'Title', 'glider-elementor' ),
                        'type'      => Controls_Manager::TEXTAREA,
                        'dynamic'   => ['active' => true],
                        'default'   => esc_html__( 'Slide Title', 'glider-elementor' ),
                        'label_block' => true,
                        'condition' => [
                            'content_from' => 'title',
                        ]
                    ]
                );

                # HTML Tag
                $items->add_control(
                    'title_tag',
                    [
                        'label' => esc_html__( 'Title Tag', 'glider-elementor' ),
                        'type' => Controls_Manager::SELECT,
                        'options' => [
                            'h1' => 'H1',
                            'h2' => 'H2',
                            'h3' => 'H3',
                            'h4' => 'H4',
                            'h5' => 'H5',
                            'h6' => 'H6',
                            'div' => 'div',
                            'span' => 'span',
                            'p' => 'p',
                        ],
                        'default' => 'h3',
                        'separator' => 'after',
                        'condition' => [
                            'content_from' => 'title',
                        ]
                    ]
                );

                # Content
                $items->add_control(
                    'content',
                    [
                        'label'         => esc_html__( 'Content', 'glider-elementor' ),
                        'type'          => Controls_Manager::WYSIWYG,
                        'dynamic'       => ['active' => true],
                        'label_block'   => true,
                        'default'       => '',
                        'placeholder'   => esc_html__( 'Enter content for item', 'glider-elementor' ),
                        'condition' => [
                            'content_from' => 'custom',
                        ]
                    ]
                );

                # Link
                $items->add_control(
                    'link',
                    [
                        'label' => esc_html__( 'Link', 'glider-elementor' ),
                        'type' => Controls_Manager::URL,
                        'dynamic' => ['active' => true],
                        'placeholder' => esc_html__( 'https://your-link.com', 'glider-elementor' ),
                        'label_block'   => true,
                        'separator' => 'before',
                    ]
                );

                # Items
                $this->add_control(
                    'items',
                    [
                        'label'         => '',
                        'type'          => Controls_Manager::REPEATER,
                        'fields'        => $items->get_controls(),
                        'default'       => [
                            [
                                'title' => esc_html__( 'Sneakers', 'glider-elementor' ),
                                'content' => esc_html__( 'Suspendisse interdum consectetur libero id faucibus nisl.', 'glider-elementor' ),
                            ],[
                                'title' => esc_html__( 'Portfolio App', 'glider-elementor' ),
                                'content' => esc_html__( 'Tristique senectus et netus et malesuada fames ac turpis.', 'glider-elementor' ),
                            ],[
                                'title' => esc_html__( 'Typography', 'glider-elementor' ),
                                'content' => esc_html__( 'Quisque non tellus orci ac auctor augue mauris.', 'glider-elementor' ),
                            ]
                        ],
                        'title_field'   => '{{ title }}'
                    ]
                );

                # Random Order
                $this->add_control(
                    'random',
                    [
                        'label' => esc_html__( 'Random Order', 'glider-elementor' ),
                        'type' => Controls_Manager::SWITCHER,
                    ]
                );

                $this->end_controls_section();

            }

        } // END Content Tab.

        /** Style Tab */
        if ( true ) {

            /** Card Section. */
            if ( true ) {

                $this->start_controls_section( 'section_card_style', [ 'label' => esc_html__( 'Card', 'glider-elementor' ), 'tab' => Controls_Manager::TAB_STYLE, 'condition' => ['skin' => 'cards'] ] );

                # Card Background
                $this->add_group_control(
                    Group_Control_Background::get_type(),
                    [
                        'name'              => 'card_background',
                        'types'             => [ 'classic', 'gradient' ],
                        'selector'          => '{{WRAPPER}} .mdp-card-default',
                    ]
                );

                # Card Border
                $this->add_group_control(
                    Group_Control_Border::get_type(),
                    [
                        'name' => 'card_border',
                        'selector' => '{{WRAPPER}} .mdp-card-default',
                        'separator' => 'before',
                    ]
                );

                # Card Border Radius
                $this->add_control(
                    'card_radius',
                    [
                        'label' => __( 'Border Radius', 'glider-elementor' ),
                        'type' => Controls_Manager::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors' => [
                            '{{WRAPPER}} .mdp-card-default' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ],
                        'separator' => 'after',
                    ]
                );

                # Card Box Shadow
                $this->add_group_control(
                    Group_Control_Box_Shadow::get_type(),
                    [
                        'name' => 'card_box_shadow',
                        'selector' => '{{WRAPPER}} .mdp-card-default',
                    ]
                );

                $this->end_controls_section();

            }

            /** Title Section. */
            if ( true ) {

                $this->start_controls_section( 'section_title_style', [ 'label' => esc_html__( 'Title', 'glider-elementor' ), 'tab' => Controls_Manager::TAB_STYLE ] );

                # Title Color
                $this->add_control(
                    'title_color',
                    [
                        'label'     => esc_html__( 'Color', 'glider-elementor' ),
                        'type'      => Controls_Manager::COLOR,
                        'scheme'    => [
                            'type'  => Color::get_type(),
                            'value' => Color::COLOR_1,
                        ],
                        'selectors' => [
                            /** Stronger selector to avoid section style from overwriting. */
                            "{{WRAPPER}} .mdp-slider-container .mdp-glider-elementor-title" => 'color: {{VALUE}};',
                        ]
                    ]
                );

                # Title Typography
                $this->add_group_control(
                    Group_Control_Typography::get_type(),
                    [
                        'name' => 'title_typography',
                        'scheme' => Typography::TYPOGRAPHY_1,
                        'selector' => "{{WRAPPER}} .mdp-slider-container .mdp-glider-elementor-title",
                    ]
                );

                # Title alignment
                $this->add_responsive_control(
                    'title_align',
                    [
                        'label'     => esc_html__( 'Alignment', 'glider-elementor' ),
                        'type'      => Controls_Manager::CHOOSE,
                        'options' => [
                            'left' => [
                                'title' => __( 'Left', 'glider-elementor' ),
                                'icon' => 'eicon-text-align-left',
                            ],
                            'center' => [
                                'title' => __( 'Center', 'glider-elementor' ),
                                'icon' => 'eicon-text-align-center',
                            ],
                            'right' => [
                                'title' => __( 'Right', 'glider-elementor' ),
                                'icon' => 'eicon-text-align-right',
                            ],
                        ],
                        'default'   => '',
                        'selectors' => [
                            '{{WRAPPER}} .mdp-glider-elementor-title' => 'text-align: {{value}} !important',
                        ],
                        'toggle'    => true,
                    ]
                );

                # Title Text Shadow
                $this->add_group_control(
                    Group_Control_Text_Shadow::get_type(),
                    [
                        'name' => 'title_text_shadow',
                        'selector' => "{{WRAPPER}} .mdp-slider-container .mdp-glider-elementor-title",
                    ]
                );

                # Title Margin
                $this->add_responsive_control(
                    'title_margin',
                    [
                        'label'         => esc_html__( 'Margin', 'glider-elementor' ),
                        'type'          => Controls_Manager::DIMENSIONS,
                        'size_units'    => [ 'px', 'em', '%' ],
                        'selectors'     => [
                            '{{WRAPPER}} .mdp-glider-elementor-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]
                );

                # Title Padding
                $this->add_responsive_control(
                    'title_padding',
                    [
                        'label'         => esc_html__( 'Padding', 'glider-elementor' ),
                        'type'          => Controls_Manager::DIMENSIONS,
                        'size_units'    => [ 'px', 'em', '%' ],
                        'selectors'     => [
                            '{{WRAPPER}} .mdp-glider-elementor-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ],
                        'separator' => 'after'
                    ]
                );

                # Title Background
                $this->add_group_control(
                    Group_Control_Background::get_type(),
                    [
                        'name'              => 'title_background',
                        'types'             => [ 'classic', 'gradient' ],
                        'selector'          => '{{WRAPPER}} .mdp-glider-elementor-title',
                    ]
                );

                # Title Border
                $this->add_group_control(
                    Group_Control_Border::get_type(),
                    [
                        'name' => 'title_border',
                        'selector' => '{{WRAPPER}} .mdp-glider-elementor-title',
                        'separator' => 'before',
                    ]
                );

                # Title Border Radius
                $this->add_control(
                    'title_radius',
                    [
                        'label' => __( 'Border Radius', 'glider-elementor' ),
                        'type' => Controls_Manager::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors' => [
                            '{{WRAPPER}} .mdp-glider-elementor-title' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ],
                        'separator' => 'after',
                    ]
                );

                $this->end_controls_section();

            }

            /** Content Section. */
            if ( true ) {

                $this->start_controls_section( 'section_content_style', [ 'label' => esc_html__( 'Content', 'glider-elementor' ), 'tab' => Controls_Manager::TAB_STYLE ] );

                # Content Color
                $this->add_control(
                    'content_color',
                    [
                        'label'     => esc_html__( 'Color', 'glider-elementor' ),
                        'type'      => Controls_Manager::COLOR,
                        'scheme'    => [
                            'type'  => Color::get_type(),
                            'value' => Color::COLOR_1,
                        ],
                        'selectors' => [
                            "{{WRAPPER}} .mdp-slider-container .mdp-glider-elementor-content" => 'color: {{VALUE}};',
                        ]
                    ]
                );

                # Content Typography
                $this->add_group_control(
                    Group_Control_Typography::get_type(),
                    [
                        'name' => 'content_typography',
                        'scheme' => Typography::TYPOGRAPHY_1,
                        'selector' => "{{WRAPPER}} .mdp-slider-container .mdp-glider-elementor-content",
                    ]
                );

                # Alignment
                $this->add_responsive_control(
                    'content_alignment',
                    [
                        'label'     => esc_html__( 'Content Alignment ', 'glider-elementor' ),
                        'type'      => Controls_Manager::CHOOSE,
                        'options' => [
                            'left' => [
                                'title' => __( 'Left', 'glider-elementor' ),
                                'icon' => 'eicon-text-align-left',
                            ],
                            'center' => [
                                'title' => __( 'Center', 'glider-elementor' ),
                                'icon' => 'eicon-text-align-center',
                            ],
                            'right' => [
                                'title' => __( 'Right', 'glider-elementor' ),
                                'icon' => 'eicon-text-align-right',
                            ],
                        ],
                        'default'   => '',
                        'selectors' => [
                            '{{WRAPPER}} .mdp-glider-elementor-content, {{WRAPPER}} .mdp-glider-elementor-content *' => 'text-align: {{value}} !important',
                        ],
                        'toggle'    => true,
                    ]
                );

                # Content Text Shadow
                $this->add_group_control(
                    Group_Control_Text_Shadow::get_type(),
                    [
                        'name' => 'content_text_shadow',
                        'selector' => "{{WRAPPER}} .mdp-slider-container .mdp-glider-elementor-content",
                    ]
                );

                # Content Margin
                $this->add_responsive_control(
                    'content_margin',
                    [
                        'label'         => esc_html__( 'Margin', 'glider-elementor' ),
                        'type'          => Controls_Manager::DIMENSIONS,
                        'size_units'    => [ 'px', 'em', '%' ],
                        'selectors'     => [
                            '{{WRAPPER}} .mdp-glider-elementor-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ]
                    ]
                );

                # Content Padding
                $this->add_responsive_control(
                    'content_padding',
                    [
                        'label'         => esc_html__( 'Padding', 'glider-elementor' ),
                        'type'          => Controls_Manager::DIMENSIONS,
                        'size_units'    => [ 'px', 'em', '%' ],
                        'selectors'     => [
                            '{{WRAPPER}} .mdp-glider-elementor-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ],
                        'separator' => 'after'
                    ]
                );

                # Content Background
                $this->add_group_control(
                    Group_Control_Background::get_type(),
                    [
                        'name'              => 'content_background',
                        'types'             => [ 'classic', 'gradient' ],
                        'selector'          => '{{WRAPPER}} .mdp-glider-elementor-content',
                    ]
                );

                # Content Border
                $this->add_group_control(
                    Group_Control_Border::get_type(),
                    [
                        'name' => 'content_border',
                        'selector' => '{{WRAPPER}} .mdp-glider-elementor-content',
                        'separator' => 'before',
                    ]
                );

                # Content Border Radius
                $this->add_control(
                    'content_radius',
                    [
                        'label' => __( 'Border Radius', 'glider-elementor' ),
                        'type' => Controls_Manager::DIMENSIONS,
                        'size_units' => [ 'px', '%' ],
                        'selectors' => [
                            '{{WRAPPER}} .mdp-glider-elementor-content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ],
                        'separator' => 'after',
                    ]
                );

                $this->end_controls_section();

            }

            /** Columns Section. */
            if ( true ) {

                $this->start_controls_section( 'section_columns', ['label' => esc_html__( 'Columns', 'glider-elementor' ), 'tab' => Controls_Manager::TAB_STYLE] );

                # Gutter
                $this->add_control(
                    'gutter',
                    [
                        'label' => esc_html__( 'Gutter', 'glider-elementor' ),
                        'type'  => Controls_Manager::SELECT,
                        'options' => [
                            'default'   => esc_html__( 'Default', 'glider-elementor' ),
                            'collapse'  => esc_html__( 'Collapse', 'glider-elementor' ),
                            'small'     => esc_html__( 'Small', 'glider-elementor' ),
                            'medium'    => esc_html__( 'Medium', 'glider-elementor' ),
                            'large'     => esc_html__( 'Large', 'glider-elementor' ),
                        ],
                        'default' => 'default',
                        'separator' => 'none'
                    ]
                );

                # Phone Portrait
                $this->add_control(
                    'phone_portrait',
                    [
                        'label' => esc_html__( 'Phone Portrait', 'glider-elementor' ),
                        'type'  => Controls_Manager::SELECT,
                        'options' => [
                            '1' => esc_html__( '1', 'glider-elementor' ),
                            '2' => esc_html__( '2', 'glider-elementor' ),
                            '3' => esc_html__( '3', 'glider-elementor' ),
                            '4' => esc_html__( '4', 'glider-elementor' ),
                            '5' => esc_html__( '5', 'glider-elementor' ),
                            '6' => esc_html__( '6', 'glider-elementor' ),
                        ],
                        'default' => '1',
                        'separator' => 'none',
                        'label_block' => true
                    ]
                );

                # Phone Landscape
                $this->add_control(
                    'phone_landscape',
                    [
                        'label' => esc_html__( 'Phone Landscape', 'glider-elementor' ),
                        'type'  => Controls_Manager::SELECT,
                        'options' => [
                            '0' => esc_html__( 'Inherit from smaller', 'glider-elementor' ),
                            '1' => esc_html__( '1', 'glider-elementor' ),
                            '2' => esc_html__( '2', 'glider-elementor' ),
                            '3' => esc_html__( '3', 'glider-elementor' ),
                            '4' => esc_html__( '4', 'glider-elementor' ),
                            '5' => esc_html__( '5', 'glider-elementor' ),
                            '6' => esc_html__( '6', 'glider-elementor' ),
                        ],
                        'default' => '0',
                        'description' => esc_html__( 'Affects device widths of 640px and larger.', 'glider-elementor' ),
                        'separator' => 'none',
                        'label_block' => true
                    ]
                );

                # Tablet
                $this->add_control(
                    'tablet',
                    [
                        'label' => esc_html__( 'Tablet', 'glider-elementor' ),
                        'type'  => Controls_Manager::SELECT,
                        'options' => [
                            '0' => esc_html__( 'Inherit from smaller', 'glider-elementor' ),
                            '1' => esc_html__( '1', 'glider-elementor' ),
                            '2' => esc_html__( '2', 'glider-elementor' ),
                            '3' => esc_html__( '3', 'glider-elementor' ),
                            '4' => esc_html__( '4', 'glider-elementor' ),
                            '5' => esc_html__( '5', 'glider-elementor' ),
                            '6' => esc_html__( '6', 'glider-elementor' ),
                        ],
                        'default' => '3',
                        'description' => esc_html__( 'Affects device widths of 960px and larger.', 'glider-elementor' ),
                        'separator' => 'none',
                        'label_block' => true
                    ]
                );

                # Desktop
                $this->add_control(
                    'desktop',
                    [
                        'label' => esc_html__( 'Desktop', 'glider-elementor' ),
                        'type'  => Controls_Manager::SELECT,
                        'options' => [
                            '0' => esc_html__( 'Inherit from smaller', 'glider-elementor' ),
                            '1' => esc_html__( '1', 'glider-elementor' ),
                            '2' => esc_html__( '2', 'glider-elementor' ),
                            '3' => esc_html__( '3', 'glider-elementor' ),
                            '4' => esc_html__( '4', 'glider-elementor' ),
                            '5' => esc_html__( '5', 'glider-elementor' ),
                            '6' => esc_html__( '6', 'glider-elementor' ),
                        ],
                        'default' => '0',
                        'description' => esc_html__( 'Affects device widths of 1200px and larger.', 'glider-elementor' ),
                        'separator' => 'none',
                        'label_block' => true
                    ]
                );

                # Large Screens
                $this->add_control(
                    'large_screens',
                    [
                        'label' => esc_html__( 'Large Screens', 'glider-elementor' ),
                        'type'  => Controls_Manager::SELECT,
                        'options' => [
                            '0' => esc_html__( 'Inherit from smaller', 'glider-elementor' ),
                            '1' => esc_html__( '1', 'glider-elementor' ),
                            '2' => esc_html__( '2', 'glider-elementor' ),
                            '3' => esc_html__( '3', 'glider-elementor' ),
                            '4' => esc_html__( '4', 'glider-elementor' ),
                            '5' => esc_html__( '5', 'glider-elementor' ),
                            '6' => esc_html__( '6', 'glider-elementor' ),
                        ],
                        'default' => '0',
                        'description' => esc_html__( 'Affects device widths of 1600px and larger.', 'glider-elementor' ),
                        'separator' => 'none',
                        'label_block' => true
                    ]
                );

                $this->end_controls_section();

            } // END Columns Section.

            /** Navigation Section. */
            if ( true ) {

                $this->start_controls_section( 'section_navigation', ['label' => esc_html__( 'Navigation', 'glider-elementor' ), 'tab' => Controls_Manager::TAB_STYLE] );

                # Slidenav Header
                $this->add_control(
                    'slidenav_options',
                    [
                        'label' => esc_html__( 'Slidenav Options', 'glider-elementor' ),
                        'type' => Controls_Manager::HEADING,
                        'separator' => 'none',
                    ]
                );

                # Slidenav
                $this->add_control(
                    'slidenav',
                    [
                        'label' => esc_html__( 'Slidenav', 'glider-elementor' ),
                        'type'  => Controls_Manager::SELECT,
                        'options' => [
                            'none' => esc_html__( 'None', 'glider-elementor' ),
                            'default' => esc_html__( 'Default', 'glider-elementor' ),
                            'top-left' => esc_html__( 'Top/Left', 'glider-elementor' ),
                            'top-right' => esc_html__( 'Top/Right', 'glider-elementor' ),
                            'bottom-left' => esc_html__( 'Bottom/Left', 'glider-elementor' ),
                            'bottom-right' => esc_html__( 'Bottom/Right', 'glider-elementor' ),
                        ],
                        'default' => 'default',
                        'separator' => 'none'
                    ]
                );

                $this->add_control(
                    'slidenav_prev_icon',
                    [
                        'label' => esc_html__( 'Previous Icon', 'glider-elementor' ),
                        'type' => Controls_Manager::ICONS,
                        'default' => [
                            'value' => 'fas fa-chevron-left',
                            'library' => 'solid',
                        ],
                    ]
                );

                $this->add_control(
                    'slidenav_next_icon',
                    [
                        'label' => esc_html__( 'Next Icon', 'glider-elementor' ),
                        'type' => Controls_Manager::ICONS,
                        'default' => [
                            'value' => 'fas fa-chevron-right',
                            'library' => 'solid',
                        ],
                    ]
                );

                # Always Shown
                $this->add_control(
                    'slidenav_always_shown',
                    [
                        'label' => esc_html__( 'Always Shown', 'glider-elementor' ),
                        'type' => Controls_Manager::SWITCHER,
                        'condition' => [
                            'slidenav!' => 'none',
                        ],
                    ]
                );

                # Translate
                $this->translate_control( 'slidenav', '.mdp-slidenav' );

                # Size
                $this->add_responsive_control(
                    'slidenav_size',
                    [
                        'label' => esc_html__( 'Size', 'glider-elementor' ),
                        'type' => Controls_Manager::SLIDER,
                        'default' => [
                            'size' => 14,
                        ],
                        'range' => [
                            'px' => [
                                'min' => 10,
                                'max' => 100,
                            ],
                        ],
                        'selectors' => [
                            '{{WRAPPER}} a.mdp-slidenav > i' => 'font-size: {{SIZE}}{{UNIT}};',
                        ],
                        'condition' => [
                            'slidenav!' => 'none',
                        ],
                    ]
                );

                # Padding
                $this->add_responsive_control(
                    'slidenav_padding',
                    [
                        'label'         => esc_html__( 'Padding', 'glider-elementor' ),
                        'type'          => Controls_Manager::DIMENSIONS,
                        'size_units'    => [ 'px', 'em', '%' ],
                        'selectors'     => [
                            '{{WRAPPER}} .mdp-icon.mdp-slidenav' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ],
                        'condition' => [
                            'slidenav!' => 'none',
                        ],
                    ]
                );

                # Margin
                $this->add_responsive_control(
                    'slidenav_margin',
                    [
                        'label'         => esc_html__( 'Margin', 'glider-elementor' ),
                        'type'          => Controls_Manager::DIMENSIONS,
                        'size_units'    => [ 'px', 'em', '%' ],
                        'selectors'     => [
                            '{{WRAPPER}} .mdp-icon.mdp-slidenav' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                        ],
                        'condition' => [
                            'slidenav!' => 'none',
                        ],
                    ]
                );

                # Arrow tabs
                $this->register_arrows_tabs();

                # Dotnav Header
                $this->add_control(
                    'dotnav_options',
                    [
                        'label' => esc_html__( 'Dotnav Options', 'glider-elementor' ),
                        'type' => Controls_Manager::HEADING,
                        'separator' => 'before',
                    ]
                );

                # Dotnav
                $this->add_control(
                    'dotnav',
                    [
                        'label' => esc_html__( 'Dotnav', 'glider-elementor' ),
                        'type'  => Controls_Manager::SELECT,
                        'options' => [
                            'none' => esc_html__( 'None', 'glider-elementor' ),
                            'default' => esc_html__( 'Default', 'glider-elementor' ),
                            'top-left' => esc_html__( 'Top/Left', 'glider-elementor' ),
                            'top-right' => esc_html__( 'Top/Right', 'glider-elementor' ),
                            'bottom-left' => esc_html__( 'Bottom/Left', 'glider-elementor' ),
                            'bottom-right' => esc_html__( 'Bottom/Right', 'glider-elementor' ),
                        ],
                        'default' => 'none',
                        'separator' => 'none'
                    ]
                );

                # Translate
                $this->translate_control( 'dotnav', '.mdp-slider-nav.mdp-dotnav' );

                # Size
                $this->add_responsive_control(
                    'dotnav_size',
                    [
                        'label' => esc_html__( 'Size', 'glider-elementor' ),
                        'type' => Controls_Manager::SLIDER,
                        'default' => [
                            'dotnav_size' => 10,
                        ],
                        'range' => [
                            'px' => [
                                'min' => 10,
                                'max' => 100,
                            ],
                        ],
                        'selectors' => [
                            '{{WRAPPER}} .mdp-dotnav > * > *' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                        ],
                        'condition' => [
                            'dotnav!' => 'none',
                        ],
                    ]
                );

                # Border width
                $this->add_responsive_control(
                    'slidenav_border_width',
                    [
                        'label' => esc_html__( 'Border width', 'glider-elementor' ),
                        'type' => Controls_Manager::SLIDER,
                        'default' => [
                            'size' => 2,
                        ],
                        'range' => [
                            'px' => [
                                'min' => 0,
                                'max' => 20,
                            ],
                        ],
                        'selectors' => [
                        ],
                        'condition' => [
                            'dotnav!' => 'none',
                        ],
                    ]
                );

                $this->register_dots_tabs();

                $this->end_controls_section();

            } // END Navigation Section.

            /** Height Section. */
            if ( true ) {

                $this->start_controls_section( 'section_height', ['label' => esc_html__( 'Height', 'glider-elementor' ), 'tab' => Controls_Manager::TAB_STYLE] );

                # Height
                $this->add_control(
                    'height',
                    [
                        'label' => esc_html__( 'Height', 'glider-elementor' ),
                        'type'  => Controls_Manager::SELECT,
                        'options' => [
                            'auto'      => esc_html__( 'Auto', 'glider-elementor' ),
                            'full'      => esc_html__( 'Viewport', 'glider-elementor' ),
                            'custom'    => esc_html__( 'Custom', 'glider-elementor' ),
                        ],
                        'description'   => esc_html__( 'The height will adapt automatically based on its content. Alternatively, the height can adapt to the height of the viewport.', 'glider-elementor' ),
                        'default'       => 'auto',
                        'separator'     => 'none'
                    ]
                );

                # Custom Height
                $this->add_control(
                    'custom_height',
                    [
                        'label' => esc_html__( 'Custom Height (viewport - value)', 'glider-elementor' ),
                        'type' => Controls_Manager::SLIDER,
                        'size_units' => [ 'px', '%' ],
                        'range' => [
                            'px' => [
                                'min' => 100,
                                'max' => 1000,
                                'step' => 50,
                            ],
                            '%' => [
                                'min' => 10,
                                'max' => 90,
                                'step' => 1,
                            ],
                        ],
                        'default' => [
                            'size' => 20,
                            'unit' => '%',
                        ],
                        'condition' => [
                            'height' => 'custom',
                        ],
                    ]
                );

                # Min. Height (px)
                $this->add_control(
                    'min_height',
                    [
                        'label' => esc_html__( 'Min. Height (px)', 'glider-elementor' ),
                        'type' => Controls_Manager::SLIDER,
                        'description' => esc_html__( 'Use an optional minimum height to prevent the glider from becoming smaller than its content on small devices.', 'glider-elementor' ),
                        'size_units' => [ 'px' ],
                        'range' => [
                            'px' => [
                                'min' => 200,
                                'max' => 800,
                                'step' => 50,
                            ],
                        ],
                        'default' => [
                            'size' => 300,
                            'unit' => 'px',
                        ],
                        'condition' => [
                            'height!' => 'auto',
                        ],
                    ]
                );

                $this->end_controls_section();

            } // END Height Section.

            /** Animations Section. */
            if ( true ) {

                $this->start_controls_section( 'section_animations', ['label' => esc_html__( 'Animations', 'glider-elementor' ), 'tab' => Controls_Manager::TAB_STYLE] );

                # Slide Sets
                $this->add_control(
                    'sets',
                    [
                        'label' => esc_html__( 'Slide Sets', 'glider-elementor' ),
                        'type' => Controls_Manager::SWITCHER,
                        'description' => esc_html__( 'Enable to loop through a set of slides instead of single items.', 'glider-elementor' ),
                    ]
                );

                # Infinite
                $this->add_control(
                    'infinite',
                    [
                        'label' => esc_html__( 'Infinite', 'glider-elementor' ),
                        'type' => Controls_Manager::SWITCHER,
                        'default' => 'yes',
                        'description' => esc_html__( 'Items are looped and you can scroll endless.', 'glider-elementor' ),
                    ]
                );

                # Center
                $this->add_control(
                    'center',
                    [
                        'label' => esc_html__( 'Center', 'glider-elementor' ),
                        'type' => Controls_Manager::SWITCHER,
                        'description' => esc_html__( 'The active item is centered.', 'glider-elementor' ),
                    ]
                );

                # Autoplay
                $this->add_control(
                    'autoplay',
                    [
                        'label' => esc_html__( 'Autoplay', 'glider-elementor' ),
                        'type' => Controls_Manager::SWITCHER,
                        'description' => esc_html__( 'Enable autoplay.', 'glider-elementor' ),
                    ]
                );

                # Interval (ms)
                $this->add_control(
                    'autoplay_interval',
                    [
                        'label' => esc_html__( 'Autoplay Interval (ms)', 'glider-elementor' ),
                        'type' => Controls_Manager::SLIDER,
                        'size_units' => [ 'px' ],
                        'range' => [
                            'px' => [
                                'min' => 1000,
                                'max' => 50000,
                                'step' => 100,
                            ],
                        ],
                        'default' => [
                            'size' => 7000,
                            'unit' => 'px',
                        ],
                        'condition' => [
                            'autoplay' => 'yes',
                        ],
                    ]
                );

                # Pause
                $this->add_control(
                    'pause_on_hover',
                    [
                        'label' => esc_html__( 'Pause', 'glider-elementor' ),
                        'type' => Controls_Manager::SWITCHER,
                        'description' => esc_html__( 'Pause autoplay when hovering the slider.', 'glider-elementor' ),
                        'condition' => [
                            'autoplay' => 'yes',
                        ],
                    ]
                );


                $this->end_controls_section();

            }  // END Animations Section.



        } // END Style Tab.

    }

    /**
     * Arrow style single tab
     * @param $sufix
     */
    private function style_navigation( $sufix ) {

        # Prepare prefixes
        $_hover = ! empty( $sufix ) ? '_' . $sufix : '';
        $hover = ! empty( $sufix ) ? ':' . $sufix : '';

        # Color
        $this->add_control(
            'slidenav_color' . $_hover,
            [
                'label'     => esc_html__( 'Color', 'glider-elementor' ),
                'type'      => Controls_Manager::COLOR,
                'default'   => '#fff',
                'selectors' => [
                    '{{WRAPPER}} a.mdp-slidenav' . $hover . ' i' => 'color: {{VALUE}};'
                ],
                'condition' => [
                    'slidenav!' => 'none',
                ],
            ]
        );

        # Background color
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name'              => 'slidenav_bg_color' . $_hover,
                'types'             => [ 'classic', 'gradient' ],
                'selector'          => '{{WRAPPER}} .mdp-icon.mdp-slidenav' . $hover,
                'condition' => [
                    'slidenav!' => 'none',
                ]
            ]
        );

        # Border
        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'slidenav_radius' . $_hover,
                'selector' => '{{WRAPPER}} .mdp-icon.mdp-slidenav',
                'condition' => [
                    'slidenav!' => 'none',
                ],
            ]
        );

        # Border radius
        $this->add_control(
            'slidenav_radius' . $_hover,
            [
                'label' => __( 'Border Radius', 'glider-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-icon.mdp-slidenav' . $hover => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'slidenav!' => 'none',
                ],
            ]
        );

        # Box shadow
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'slidenav_box_shadow' . $_hover,
                'selector' => '{{WRAPPER}} .mdp-icon.mdp-slidenav' . $hover,
                'condition' => [
                    'slidenav!' => 'none',
                ],
            ]
        );

    }

    /**
     * Arrow style tabs
     */
    private function register_arrows_tabs() {

        $this->start_controls_tabs(
            'arrows_style_tabs',
            [
                'condition' => [
                    'slidenav!' => 'none',
                ],
            ]
        );

            # Normal tab
            $this->start_controls_tab(
                'arrows_style_normal_tab',
                [
                    'label' => __( 'Normal', 'glider-elementor' ),
                ]
            );

                $this->style_navigation( '' );

            $this->end_controls_tab();

            # Hover tab
            $this->start_controls_tab(
                'arrows_style_hover_tab',
                [
                    'label' => __( 'Hover', 'glider-elementor' ),
                ]
            );

                $this->style_navigation( 'hover' );

            $this->end_controls_tab();

        $this->end_controls_tabs();

    }

    /**
     * Dots style single tab
     * @param $sufix
     */
    private function style_dots( $sufix ) {

        # Prepare prefixes
        $_hover = ! empty( $sufix ) ? '_' . $sufix : '';
        $hover = ! empty( $sufix ) ? ':' . $sufix : '';

        # Color
        $this->add_control(
            'dotnav_color' . $_hover,
            [
                'label'     => esc_html__( 'Color', 'glider-elementor' ),
                'type'      => Controls_Manager::COLOR,
                'default'   => 'rgba(102, 102, 102, 0.4)',
                'selectors' => [
                    '{{WRAPPER}} .mdp-dotnav a' . $hover => 'border: {{slidenav_border_width.size}}{{slidenav_border_width.unit}} solid {{VALUE}};',
                    '{{WRAPPER}} .mdp-dotnav .mdp-active a'. $hover => 'background-color: {{VALUE}};'
                ],
                'condition' => [
                    'dotnav!' => 'none',
                ],
            ]
        );

        # Border radius
        $this->add_control(
            'dotnav_radius' . $_hover,
            [
                'label' => __( 'Border Radius', 'glider-elementor' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'default' => [
                    'size' => 50,
                    'unit' => 'px'
                ],
                'selectors' => [
                    '{{WRAPPER}} .mdp-dotnav a' . $hover => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'dotnav!' => 'none',
                ],
            ]
        );

        # Box shadow
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'dotnav_box_shadow' . $_hover,
                'selector' => '{{WRAPPER}} .mdp-dotnav a' . $hover,
                'condition' => [
                    'dotnav!' => 'none',
                ],
            ]
        );

    }

    /**
     * Dots navogation tabs
     */
    private function register_dots_tabs() {

        $this->start_controls_tabs(
            'dots_style_tabs',
            [
                'condition' => [
                    'dotnav!' => 'none',
                ],
            ]
        );

            # Normal tab
            $this->start_controls_tab(
                'dots_style_normal_tab',
                [
                    'label' => __( 'Normal', 'glider-elementor' ),
                ]
            );

                $this->style_dots( '' );

            $this->end_controls_tab();

            # Hover tab
            $this->start_controls_tab(
                'dots_style_hover_tab',
                [
                    'label' => __( 'Hover', 'glider-elementor' ),
                ]
            );

                $this->style_dots( 'hover' );

            $this->end_controls_tab();

        $this->end_controls_tabs();

    }

    /**
     * Navogation translate contorls
     * @param $prefix
     * @param $selector
     */
    private function translate_control( $prefix, $selector ) {

        $range = [
            'px' => [
                'min' => -100,
                'max' => 100,
                'step' => 1,
            ],
            '%' => [
                'min' => -100,
                'max' => 100,
            ],
        ];

        $default = [
            'value' => 0,
            'unit' => 'px',
        ];

        $this->add_responsive_control(
            $prefix . '_translate_x',
            [
                'label' => esc_html__( 'Translate X', 'glider-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => $range,
                'default' => $default,
                'selectors' => [
                    '{{WRAPPER}} ' . $selector => 'transform: translate( {{SIZE}}{{UNIT}}, {{' . $prefix . '_translate_y.SIZE}}{{' . $prefix . '_translate_y.UNIT}} );',
                ],
                'condition' => [
                    $prefix . '!' => 'none',
                ],
            ]
        );

        $this->add_responsive_control(
            $prefix . '_translate_y',
            [
                'label' => esc_html__( 'Translate Y', 'glider-elementor' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px', '%' ],
                'range' => $range,
                'default' => $default,
                'selectors' => [
                    '{{WRAPPER}} ' . $selector => 'transform: translate( {{' . $prefix . '_translate_x.SIZE}}{{' . $prefix . '_translate_x.UNIT}}, {{SIZE}}{{UNIT}} );',
                ],
                'condition' => [
                    $prefix . '!' => 'none',
                ],
            ]
        );

    }

    /**
     * Render Frontend Output. Generate the final HTML on the frontend.
     *
     * @since 1.0.0
     * @access protected
     **/
    protected function render() {

        /** Get Widget Settings. */
        $settings = $this->get_settings_for_display();

        /** No items = No render. */
        if ( ! count( $settings['items'] ) ) { return; }

        /** Random Order. */
        if ( 'yes' === $settings['random'] ) { shuffle( $settings['items'] ); }

        /** Slider classes. */
        $classes = $this->get_slider_classes( $settings );

        /** Slider Parameters. */
        $params = $this->get_slider_params( $settings );

        /** Slider Height. */
        $height = $this->get_slider_height( $settings );

        ?>
        <!-- Start Glider for Elementor WordPress Plugin -->
        <?php if ( Plugin::$instance->editor->is_edit_mode() ) : ?>
            <script>window.mdpGliderElementorEditor = true;</script>
        <?php endif; ?>
        <div class="mdp-glider-elementor-box mdp-position-relative mdp-visible-toggle" mdp-slider="<?php esc_attr_e( $params ); ?>">

            <div class="mdp-position-relative">
                <ul class="<?php esc_attr_e( $classes ); ?>"

                    <?php if ( 'cards' === $settings['skin'] ) : ?>
                        mdp-height-match="target: .mdp-card"
                    <?php endif; ?>

                    <?php if ( 'auto' !== $settings['height'] ) : ?>
                        mdp-height-viewport="<?php esc_html_e( $height ); ?>"
                    <?php endif; ?>
                >
                    <?php foreach ( $settings['items'] as $index => $item ) : ?>
                        <li>
                            <?php if ( 'overlay' === $settings['skin'] ) : ?>
                                <?php $this->render_overlay_skin( $settings, $item ); ?>
                            <?php elseif ( 'cards' === $settings['skin'] ) : ?>
                                <?php $this->render_cards_skin( $settings, $item ); ?>
                            <?php endif; ?>
                        </li>
                    <?php endforeach; ?>
                </ul>

                <?php $this->render_slidenav( $settings ); // Render arrows. ?>

            </div>

            <?php $this->render_dotnav( $settings ); // Render dots. ?>

        </div>
        <!-- End Glider for Elementor WordPress Plugin -->
        <?php

    }

    private function get_slider_height( $settings ) {

        $height = '';

        /** Min. Height. */
        if ( ! empty( $settings['min_height']['size'] ) ) { $settings['min_height']['size'] = '300'; }

        if ( 'full' === $settings['height'] ) {
            $height = 'offset-top: true; minHeight: ' . $settings['min_height']['size'] . ';';
        } elseif ( 'custom' === $settings['height'] ) {

            $unit = '';
            if ( '%' !== $settings['custom_height']['unit'] ) { $unit = 'px'; }

            $height = 'offset-top: true; minHeight: ' . $settings['min_height']['size'] . '; offset-bottom: ' . $settings['custom_height']['size'] . $unit . ';';
        }

        return $height;
    }

    private function render_cards_skin( $settings, $item ) {
        ?>
        <div class="mdp-cards-skin mdp-position-relative mdp-card mdp-card-default">

            <?php if ( in_array( $settings['image_position'], ['left', 'right'] ) ) : ?>

                <div class="mdp-grid-collapse mdp-child-width-1-2@s mdp-margin" mdp-grid="">

                    <?php if ( 'left' === $settings['image_position'] ) : ?>
                        <?php $this->mdp_render_image( $item, 'left' ); ?>
                    <?php endif; ?>

                    <div>
                        <div class="mdp-card-body">
                            <?php $this->mdp_render_content( $item ); ?>
                        </div>
                    </div>

                    <?php if ( 'right' === $settings['image_position'] ) : ?>
                        <?php $this->mdp_render_image( $item, 'right' ); ?>
                    <?php endif; ?>

                </div>

            <?php endif; ?>

            <?php if ( in_array( $settings['image_position'], ['top', 'bottom'] ) ) : ?>

                <?php if ( 'top' === $settings['image_position'] ) : ?>
                    <?php $this->a_start( $item ); ?>
                    <div class="mdp-card-media-top">
                        <?php echo Group_Control_Image_Size::get_attachment_image_html( $item ); ?>
                    </div>
                    <?php $this->a_end( $item ); ?>
                <?php endif; ?>

                <div class="mdp-card-body">
                    <?php $this->mdp_render_content( $item ); ?>
                </div>

                <?php if ( 'bottom' === $settings['image_position'] ) : ?>
                    <?php $this->a_start( $item ); ?>
                    <div class="mdp-card-media-bottom">
                        <?php echo Group_Control_Image_Size::get_attachment_image_html( $item ); ?>
                    </div>
                    <?php $this->a_end( $item ); ?>
                <?php endif; ?>

            <?php endif; ?>

            <?php if ( 'none' === $settings['image_position'] ) : ?>
                <div class="mdp-card-body">
                    <?php $this->mdp_render_content( $item ); ?>
                </div>
            <?php endif; ?>

        </div>
        <?php
    }

    private function render_overlay_skin( $settings, $item ) {
        ?>
        <?php $this->a_start( $item ); ?>

        <div class="mdp-cover-container">
            <img src="<?php esc_attr_e( $item['image']['url'] ); ?>"
                 alt=""
                <?php if ( 'auto' !== $settings['height'] ) : ?>
                    mdp-cover=""
                <?php endif; ?>
            >
            <div class="<?php esc_attr_e( $settings['content_position'] ); ?>">
                <div class="mdp-panel">
                    <?php $this->mdp_render_content( $item ); ?>
                </div>
            </div>
        </div>

        <?php $this->a_end( $item ); ?>

        <?php
    }

    private function a_start( $item ) {
if ( $item['link']['url'] ) :
    ?>
    <a
            href="<?php esc_attr_e( $item['link']['url'] ); ?>"
            <?php if ( $item['link']['is_external'] ) : ?>target="_blank" <?php endif; ?>
            <?php if ( $item['link']['nofollow'] ) : ?>rel="nofollow" <?php endif; ?>
    >
        <?php
        endif;
        }

    private function a_end( $item ) {

        if ( $item['link']['url'] ) :
        ?></a><?php
endif;

}

    private function mdp_render_image( $item, $align = 'left' ) {
        ?>
        <?php $this->a_start( $item ); ?>
        <div class="mdp-card-media-<?php esc_attr_e( $align ); ?> mdp-cover-container">
            <?php echo Group_Control_Image_Size::get_attachment_image_html( $item ); ?>
            <canvas width="600" height="400"></canvas>
        </div>
        <?php
        $this->a_end( $item );

    }

    private function mdp_render_content( $item ) {

        /** Content disabled. */
        if ( 'none' === $item['content_from'] ) { return; }

        $tag = 'h3';

        /** Content from attachment details. */
        if ( 'attachment' === $item['content_from'] ) {

            /** No attachment id. */
            if ( ! $item['image']['id'] ) { return; }

            /** Get content form attachment. */
            $title = get_the_title( $item['image']['id'] );
            $content = get_the_content( null, false, $item['image']['id'] );

            /** Custom Content. */
        } else {

            $title = $item['title'];
            $content = $item['content'];
            $tag = $item['title_tag'];

        }

        if ( ! empty( $title ) ) {
            echo sprintf( '<%1$s class="mdp-glider-elementor-title">%2$s</%1$s>', $tag, esc_html__( $title ) );
        }

        if ( ! empty( $content ) ) {
            ?><div class="mdp-glider-elementor-content"><?php echo wp_kses_post( $content ); ?></div><?php
        }

    }

    /**
     * Return classes for for slider.
     *
     * @param array $settings - Current widget settings.
     *
     * @since  1.0.0
     * @access protected
     *
     * @return string
     **/
    private function get_slider_classes( $settings ) {

        $classes = ' mdp-slider-items ';

        /** Space between items. */
        $classes .= ' mdp-grid mdp-grid-' . $settings['gutter'];

        /** Responsive Columns. */
        $classes .= $this->get_responsive_columns( $settings );

        /** Grid Match. */
        if ( 'auto' !== $settings['height'] ) {
            $classes .= ' mdp-grid-match ';
        }

        /** Remove multiple spaces. */
        $classes = preg_replace( '!\s+!', ' ', $classes );
        $classes = trim( $classes );

        return $classes;

    }

    /**
     * Return params string for slider.
     *
     * @param array $settings - Current widget settings.
     *
     * @since  1.0.0
     * @access protected
     *
     * @return string
     **/
    private function get_slider_params( $settings ) {

        $params = '';

        /** Slide Sets */
        if ( 'yes' === $settings['sets'] ) { $params .= 'sets: true; '; }

        /** Infinite */
        if ( 'yes' !== $settings['infinite'] ) { $params .= 'finite: true; '; }

        /** Center */
        if ( 'yes' === $settings['center'] ) { $params .= 'center: true; '; }

        /** Autoplay */
        if ( 'yes' === $settings['autoplay'] ) {
            $params .= 'autoplay: true; ';

            /** Autoplay Interval */
            $params .= 'autoplay-interval: ' . $settings['autoplay_interval']['size'] . '; ';
        }

        /** Pause on hover. */
        if ( 'yes' === $settings['pause_on_hover'] ) {
            $params .= 'pause-on-hover: true; ';
        } else {
            $params .= 'pause-on-hover: false; ';
        }

        return $params;

    }

    /**
     * Return responsive classes for slider.
     *
     * @param array $settings - Current widget settings.
     *
     * @since  1.0.0
     * @access protected
     *
     * @return string
     **/
    private function get_responsive_columns( $settings ) {

        $columns = ' mdp-child-width-1-' . $settings['phone_portrait'];

        if ( $settings['phone_landscape'] ) {
            $columns .= ' mdp-child-width-1-' . $settings['phone_landscape'] . '@s ';
        }

        if ( $settings['tablet'] ) {
            $columns .= ' mdp-child-width-1-' . $settings['tablet'] . '@m ';
        }

        if ( $settings['desktop'] ) {
            $columns .= ' mdp-child-width-1-' . $settings['desktop'] . '@l ';
        }

        if ( $settings['large_screens'] ) {
            $columns .= ' mdp-child-width-1-' . $settings['large_screens'] . '@xl ';
        }

        return $columns;
    }

    /**
     * Create a dot navigation to operate items or to scroll to different set.
     *
     * @param array $settings - Current widget settings.
     *
     * @since  1.0.0
     * @access protected
     **/
    private function render_dotnav( $settings ) {

        $dotnav = $settings['dotnav'];

        /** Do we need dotnav? */
        if ( 'none' === $dotnav ) { return; }

        /** Default position for  ○ ● ○ ○  bottom center. */
        if ( 'default' === $dotnav ) {
            ?><ul class="mdp-slider-nav mdp-dotnav mdp-flex-center mdp-margin"></ul><?php

            // mdp-dotnav-vertical
            /** Other positions. */
        } else {

            ?>
            <div class="mdp-dotnav-container mdp-position-<?php esc_attr_e( $dotnav ); ?>">
                <ul class="mdp-slider-nav mdp-dotnav mdp-flex-center"></ul>
            </div>
            <?php

        }

    }

    /**
     * Render navigation with previous and next buttons to flip through items.
     *
     * @param array $settings - Current widget settings.
     *
     * @since  1.0.0
     * @access protected
     **/
    private function render_slidenav( $settings ) {

        $slidenav = $settings['slidenav'];

        /** Do we need slidenav? */
        if ( 'none' === $slidenav ) { return; }

        /** Show on hover or always. */
        $always = ( 'yes' === $settings['slidenav_always_shown'] ) ? '' : 'mdp-hidden-hover';

        /** Default position for <- -> center left and center right. */
        if ( 'default' === $slidenav ) {

            ?>
            <a class="mdp-position-center-left mdp-position-small <?php esc_attr_e( $always ); ?>" href="#" mdp-slidenav-previous="" mdp-slider-item="previous">
                <?php Icons_Manager::render_icon( $settings[ 'slidenav_prev_icon' ], [ 'aria-hidden' => 'true' ] ); ?>
            </a>
            <a class="mdp-position-center-right mdp-position-small <?php esc_attr_e( $always ); ?>" href="#" mdp-slidenav-next="" mdp-slider-item="next">
                <?php Icons_Manager::render_icon( $settings[ 'slidenav_next_icon' ], [ 'aria-hidden' => 'true' ] ); ?>
            </a>
            <?php

            /** Other positions. */
        } else {

            ?>
            <div class="mdp-slidenav-container mdp-position-<?php esc_attr_e( $slidenav ); ?>">
                <div>
                    <a class="mdp-position-small <?php esc_attr_e( $always ); ?>" href="#" mdp-slidenav-previous="" mdp-slider-item="previous">
                        <?php Icons_Manager::render_icon( $settings[ 'slidenav_prev_icon' ], [ 'aria-hidden' => 'true' ] ); ?>
                    </a>
                </div>
                <div>
                    <a class="mdp-position-small <?php esc_attr_e( $always ); ?>" href="#" mdp-slidenav-next="" mdp-slider-item="next">
                        <?php Icons_Manager::render_icon( $settings[ 'slidenav_next_icon' ], [ 'aria-hidden' => 'true' ] ); ?>
                    </a>
                </div>
            </div>
            <?php

        }

    }

    /**
     * Return link for documentation.
     *
     * Used to add stuff after widget.
     *
     * @since 1.0.0
     * @access public
     **/
    public function get_custom_help_url() {
        return 'https://docs.merkulov.design/category/glider-elementor/';
    }

}
